<?php declare(strict_types=1);

namespace VideoAISoftware\Component\ProductAiVideo;

use Doctrine\DBAL\Connection;
use Shopware\Core\Content\Media\MediaCollection;
use Shopware\Core\Content\Media\MediaException;
use Shopware\Core\Content\Product\Aggregate\ProductMedia\ProductMediaCollection;
use Shopware\Core\Content\Product\Aggregate\ProductMedia\ProductMediaEntity;
use Shopware\Core\Content\Product\Exception\ProductNotFoundException;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\NotFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Uuid\Uuid;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use VideoAISoftware\Component\Api\MiddlewareClientFactory;
use VideoAISoftware\Component\MediaPersister;
use VideoAISoftware\Content\ProductAiVideo\ProductAiVideoDefinition;
use VideoAISoftware\Content\ProductAiVideo\ProductAiVideoEntity;
use VideoAISoftware\Content\ProductSetting\GenerateVideoState;
use VideoAISoftware\Content\ProductSetting\ProductSettingDefinition;
use VideoAISoftware\Content\ProductSetting\ProductSettingEntity;
use VideoAISoftware\Exception\InvalidVideoScriptException;
use VideoAISoftware\Exception\ParseUrlException;
use VideoAISoftware\Exception\ProductHasNoProductSettingException;
use VideoAISoftware\Exception\ProductHasNoVideoScriptException;
use VideoAISoftware\Exception\TemplateNotSetException;
use VideoAISoftware\Util\ContextHelper;
use VideoAISoftware\Util\PluginConfigService;

readonly class VideoManager
{
    public function __construct(
        private MediaPersister          $mediaPersister,
        #[Autowire(service: ProductAiVideoDefinition::ENTITY_NAME . '.repository')]
        private EntityRepository        $productAiVideoRepository,
        private MiddlewareClientFactory $middlewareClientFactory,
        private PluginConfigService     $pluginConfigService,
        #[Autowire(service: ProductSettingDefinition::ENTITY_NAME . '.repository')]
        private EntityRepository        $productSettingRepository,
        private ContextHelper           $contextHelper,
        private EntityRepository        $mediaRepository,
        private EntityRepository        $productRepository,
        private EntityRepository        $productMediaRepository,
        private Connection              $connection,
    )
    {
    }

    public function generateVideo(
        string  $productId,
        Context $context
    ): array
    {
        $product = $this->fetchProduct($productId, $context);
        /** @var ?ProductSettingEntity $productSetting */
        $productSetting = $product->getExtension(ProductSettingDefinition::ENTITY_EXTENSION);

        if (!$productSetting) {
            throw new ProductHasNoProductSettingException();
        }

        $videoScript = $productSetting->getVideoScript();
        if (!$videoScript) {
            throw new ProductHasNoVideoScriptException();
        }

        $templateId = $this->getTemplateIdFromVideoScript($videoScript);
        if (!$templateId) {
            throw new TemplateNotSetException();
        }

        $templateData = $this->formatTemplateData($videoScript);

        $productDescription = $product->getTranslation('description');

        if (mb_strlen($productDescription) > 400) {
            $productDescription = mb_substr($productDescription, 0, 400);
        }

        $client = $this->middlewareClientFactory->create();

        $this->productSettingRepository->upsert([[
            'productId' => $productId,
            'generateVideoState' => GenerateVideoState::PENDING->value,
        ]], $context);

        $response = $client->request(
            method: 'POST',
            url: 'videos',
            options: [
                'body' => [
                    'outputLanguage' => $this->pluginConfigService->getProductLanguageIsoCode(),
                    'id' => $product->getId(),
                    'templateId' => $templateId,
                    'templateData' => json_encode($templateData),
                    'title' => $product->getTranslation('name'),
                    'description' => $productDescription
                ]
            ]
        );

        $json = $response->toArray();

        return [
            'message' => $json['message']
        ];
    }

    private function fetchProduct(string $productId, Context $context): ProductEntity
    {
        $criteria = new Criteria([$productId]);
        $criteria->addAssociation(
            ProductSettingDefinition::ENTITY_EXTENSION
        );

        if ($product = $this->productRepository->search($criteria, $context)->first()) {
            return $product;
        }

        throw new ProductNotFoundException($productId);
    }

    private function formatTemplateData(array $videoScript): array
    {
        if (!array_key_exists('scenes', $videoScript)) {
            throw new InvalidVideoScriptException("Key 'scenes' does not exist");
        }

        $mediaCollection = $this->prefetchMedia($videoScript);

        $templateData = [];
        foreach ($videoScript['scenes'] as $scene) {
            $templateData["sceneScript{$scene['sceneId']}"] = $scene["sceneScriptOutputValue"] ?? null;

            if (!array_key_exists('variables', $scene)) {
                continue;
            }

            foreach ($scene['variables'] as $variable) {
                switch ($variable['variableType'] ?? null) {
                    case 'text':
                        $templateData[$variable['variableName']] = $variable['variableOutputValue'];
                        break;
                    case 'image':
                        $templateData[$variable['variableName']] = $mediaCollection
                            ->get($variable['variableOutputValue'] ?? null)
                            ?->getPath() ?? null;
                        break;
                }
            }
        }

        return $templateData;
    }

    private function prefetchMedia(array $videoScript): MediaCollection
    {
        $ids = [];

        foreach ($videoScript['scenes'] as $scene) {
            if (!array_key_exists('variables', $scene)) {
                continue;
            }

            foreach ($scene['variables'] as $variable) {
                if ($variable['variableType'] === 'image') {
                    $ids[] = $variable['variableOutputValue'] ?? null;
                }
            }
        }

        if (empty($ids)) {
            return new MediaCollection();
        }

        return $this->mediaRepository->search(
            new Criteria($ids),
            $this->contextHelper->createCLIContext()
        )->getEntities();
    }

    /**
     * @throws ParseUrlException|MediaException
     */
    public function persist(string $url, string $productId, Context $context): ProductAiVideoEntity
    {
        /** @var ?ProductEntity $product */
        $product = $this->productRepository->search(new Criteria([$productId]), $context)->first();

        if (!$product) {
            throw new ProductNotFoundException($productId);
        }

        $folderName = mb_substr($product->getName(), 0, 1 + mb_strlen($product->getId()));
        $folderName .= "-{$product->getId()}";

        $mediaId = $this->mediaPersister->createMediaFromUrl(
            url: $url,
            context: $context,
            folderName: $folderName
        );

        $result = $this->productAiVideoRepository->create([[
            'productId' => $productId,
            'mediaId' => $mediaId,
            // 'languageId' => $context->getLanguageId() // TODO: Add when language implemented in middleware
        ]], $context);

        $ids = $result->getPrimaryKeys(ProductAiVideoDefinition::ENTITY_NAME);
        $criteria = new Criteria($ids);
        return $this->productAiVideoRepository->search($criteria, $context)->first();
    }

    public function publish(string $id, Context $context, ?string $parentId = null): void
    {
        /** @var ?ProductAiVideoEntity $aiVideo */
        $aiVideo = $this->productAiVideoRepository->search(
            new Criteria([$id]),
            $context
        )->first();

        if (!$aiVideo) return;

        $productMediaId = $this->insertToProductMedia(
            $aiVideo->getProductId(),
            $aiVideo->getMediaId(),
            $context
        );

        $this->productAiVideoRepository->update([[
            'id' => $id,
            // 'languageId' => $context->getLanguageId(), // TODO: Add when language implemented in middleware
            'published' => true,
            'productMediaId' => $productMediaId,
            'productMediaVersionId' => $context->getVersionId(),
            'parentId' => $parentId
        ]], $context);

        $this->publishForChildren($aiVideo, $context);
    }

    public function unpublish(string $id, Context $context, ?bool $isChild = false): void
    {
        /** @var ?ProductAiVideoEntity $aiVideo */
        $aiVideo = $this->productAiVideoRepository->search(
            (new Criteria([$id]))->addAssociation('children'),
            $context
        )->first();

        if (!$aiVideo) return;

        // Automatically sets published = false, see ProductAiVideoUpdater
        $this->productMediaRepository->delete([[
            'id' => $aiVideo->getProductMediaId()
        ]], $context);

        // renumber productMedia

        $productMedia = $this->fetchProductMedia($aiVideo->getProductId(), $context);

        $position = 0;
        foreach ($productMedia as $productMediaEntity) {
            $productMediaEntity->setPosition($position++);
        }

        $this->productRepository->update([[
            'id' => $aiVideo->getProductId(),
            'media' => array_values($productMedia->map(fn(ProductMediaEntity $productMediaEntity) => [
                'id' => $productMediaEntity->getId(),
                'position' => $productMediaEntity->getPosition()
            ]))]], $context);

        if ($aiVideo->getChildren()->count() > 0) {
            foreach ($aiVideo->getChildren() as $child) {
                $this->unpublish($child->getId(), $context, true);
            }
        }
    }

    private function insertToProductMedia(string $productId, string $mediaId, Context $context): string
    {
        $publishedProductMediaIds = $this->fetchPublishedVideoProductMediaIds($productId, $context);

        $this->connection->beginTransaction();

        try {
            $this->productMediaRepository->delete(array_values(array_map(fn(string $id) => ['id' => $id], $publishedProductMediaIds)), $context);
            $productMedia = $this->fetchProductMedia($productId, $context);

            $collection = new ProductMediaCollection();
            $firstProductMedia = $productMedia->first();

            if ($firstProductMedia) {
                $collection->add($firstProductMedia);
                $productMedia->remove($firstProductMedia->getId());
            }

            $insertProductMedia = new ProductMediaEntity();
            $insertProductMedia->setId(Uuid::randomHex());
            $insertProductMedia->setMediaId($mediaId);
            $collection->add($insertProductMedia);

            foreach ($productMedia as $media) {
                $collection->add($media);
            }

            $position = 0;
            foreach ($collection as $media) {
                $media->setPosition($position++);
            }

            $this->productRepository->upsert([[
                'id' => $productId,
                'media' => $collection->map(fn(ProductMediaEntity $productMedia) => [
                    'id' => $productMedia->getId(),
                    'mediaId' => $productMedia->getMediaId(),
                    'position' => $productMedia->getPosition()
                ])
            ]], $context);

            $this->connection->commit();

            return $insertProductMedia->getId();
        } catch (\Throwable $throwable) {
            $this->connection->rollBack();
            throw $throwable;
        }
    }

    /**
     * @return string[]
     */
    private function fetchPublishedVideoProductMediaIds(string $productId, Context $context): array
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('productId', $productId));
        $criteria->addFilter(
            new EqualsFilter('published', true),
            new NotFilter(NotFilter::CONNECTION_AND, [
                new EqualsFilter('productMediaId', null)
            ])
        );

        $collection = $this->productAiVideoRepository->search($criteria, $context)->getEntities();
        $criteria = new Criteria();
        $criteria->addFilter(
            new EqualsAnyFilter('parentId', $collection->map(fn (ProductAiVideoEntity $entity) => $entity->getId()))
        );

        $children = $this->productAiVideoRepository->search($criteria, $context)->getEntities();

        foreach ($children as $child) {
            $collection->add($child);
        }

        return $collection
            ->map(fn(ProductAiVideoEntity $entity) => $entity->getProductMediaId());
    }

    private function fetchProductMedia(string $productId, Context $context): ProductMediaCollection
    {
        $criteria = new Criteria([$productId]);
        $criteria->addAssociation('media');

        $criteria->addSorting(
            new FieldSorting('media.position', FieldSorting::ASCENDING)
        );

        /** @var ?ProductEntity $product */
        $product = $this->productRepository->search($criteria, $context)->first();

        if (!$product) {
            throw new ProductNotFoundException($productId);
        }

        return $product->getMedia();
    }

    private function findSecondPositionOrFirst(ProductMediaCollection $productMedia): int
    {
        if ($productMedia->count() === 0) {
            return 0;
        }

        return $productMedia->first()->getPosition() + 1;
    }

    private function publishForChildren(?ProductAiVideoEntity $aiVideo, Context $context): void
    {
        $criteria = new Criteria();
        $criteria->addFilter(
            new EqualsFilter('parentId', $aiVideo->getProductId()),
        );
        $criteria->addAssociation('media');

        $childrenWithMedia = $this->productRepository
            ->search($criteria,$context)
            ->getEntities()
            ->filter(fn (ProductEntity $product) => $product->getMedia()->count() > 0);

        $childrenIds = $childrenWithMedia->map(fn (ProductEntity $product) => $product->getId());

        if (empty($childrenIds)) return;

        // create aiVideo entry for each child
        $result = $this->productAiVideoRepository->create(array_values(array_map(fn (string $id) => [
            'productId' => $id,
            'mediaId' => $aiVideo->getMediaId()
        ], $childrenIds)), $context);

        $ids = $result->getPrimaryKeys(ProductAiVideoDefinition::ENTITY_NAME);

        if (!is_array($ids)) {
            $ids = [$ids];
        }

        foreach ($ids as $id) {
            $this->publish($id, $context, $aiVideo->getId());
        }
    }

    private function getTemplateIdFromVideoScript(array $videoScript): ?string
    {
        if (!array_key_exists('meta', $videoScript)) return null;
        if (!array_key_exists('templateId', $videoScript['meta'])) return null;

        return $videoScript['meta']['templateId'];
    }
}
