<?php declare(strict_types=1);

namespace VideoAISoftware\Component;

use Shopware\Core\Content\Media\File\FileSaver;
use Shopware\Core\Content\Media\File\MediaFile;
use Shopware\Core\Content\Media\MediaException;
use Shopware\Core\Content\Media\MediaService;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Uuid\Uuid;
use Symfony\Component\HttpFoundation\Request;
use VideoAISoftware\Content\ProductAiVideo\ProductAiVideoDefinition;
use VideoAISoftware\Exception\ParseUrlException;

class MediaPersister
{
    public function __construct(
        protected readonly MediaService $mediaService,
        protected FileSaver             $fileSaver,
        protected EntityRepository      $mediaRepository,
        protected EntityRepository      $mediaFolderRepository,
    )
    {
    }

    /**
     * @throws \Exception
     */
    public function persistMedia(
        MediaFile $mediaFile,
        string    $destination,
        Context   $context,
        string    $folderName,
        bool      $private = false,
    ): string
    {
        $aiVideoFolder = $this->getMediaDefaultFolderId(ProductAiVideoDefinition::ENTITY_NAME, $context);

        $folderName = mb_substr($folderName, 0, 255);
        $folderId = $this->getOrCreateFolder($folderName, $aiVideoFolder, $context);

        $mediaId = $this->createMediaInFolder($folderId, $context, $private);

        try {
            $this->fileSaver->persistFileToMedia(
                $mediaFile,
                $destination,
                $mediaId,
                $context
            );
        } catch (MediaException $e) {
            $this->mediaRepository->delete([[
                'id' => $mediaId
            ]], $context);

            throw $e;
        }

        return $mediaId;
    }

    /**
     * @throws ParseUrlException
     */
    public function createMediaFromUrl(
        string  $url,
        Context $context,
        string  $folderName,
        array   $customFields = [],
        bool    $private = false,
    ): string
    {
        $parsed = parse_url($url);
        if (!$parsed) {
            throw new ParseUrlException($url);
        }

        $pathInfo = pathinfo($parsed['path'] ?? '');

        $mediaFile = $this->fetchFileFromUrl($url, $pathInfo['extension'] ?? '');

        $destination = $this->destinationFromPathInfo($pathInfo);
        if (str_starts_with($destination, '_')) {
            $destination = substr($destination, 1);
        }

        $mediaId = $this->persistMedia($mediaFile, $destination, $context, $folderName, $private);
        if (!$customFields) return $mediaId;

        $this->mediaRepository->update([[
            'id' => $mediaId,
            'customFields' => $customFields
        ]], $context);

        return $mediaId;
    }

    public function destinationFromPathInfo(array $pathInfo): string
    {
        $destination = urldecode($pathInfo['dirname']) . '/' . urldecode($pathInfo['filename']);
        return preg_replace('/[^a-zA-Z0-9_]/', '_', $destination) . '-' . Uuid::randomHex();
    }

    public function fetchFileFromUrl(string $url, mixed $extension): ?MediaFile
    {
        $request = new Request();
        $request->query->set('url', $url);
        $request->query->set('extension', $extension);
        $request->request->set('url', $url);
        $request->request->set('extension', $extension);
        $request->headers->set('content-type', 'application/json');

        $file = $this->mediaService->fetchFile($request);
        if ($file->getFileSize() > 0) {
            return $file;
        }

        return null;
    }

    private function getMediaDefaultFolderId(string $entity, Context $context): ?string
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('media_folder.defaultFolder.entity', $entity));
        $criteria->addAssociation('defaultFolder');
        $criteria->setLimit(1);

        if ($id = $this->mediaFolderRepository->searchIds($criteria, $context)->firstId()) {
            return $id;
        }

        throw new \Exception("Folder for entity '$entity' not found.");
    }

    private function getOrCreateFolder(string $folderName, string $parentFolderId, Context $context): string
    {
        $criteria = new Criteria();
        $criteria->addFilter(
            new EqualsFilter('parentId', $parentFolderId),
            new EqualsFilter('name', $folderName)
        );

        if ($id = $this->mediaFolderRepository->searchIds($criteria, $context)->firstId()) {
            return $id;
        }

        $id = Uuid::randomHex();

        $this->mediaFolderRepository->create([[
            'id' => $id,
            'parentId' => $parentFolderId,
            'name' => $folderName,
            'configuration' => [
                'createThumbnails' => false,
                'keepAspectRatio' => false,
                'thumbnailQuality' => 80,
                'private' => false,
                'noAssociation' => false
            ]
        ]], $context);

        return $id;
    }

    private function createMediaInFolder(string $folderId, Context $context, bool $private): string
    {
        $mediaId = Uuid::randomHex();
        $this->mediaRepository->create(
            [
                [
                    'id' => $mediaId,
                    'private' => $private,
                    'mediaFolderId' => $folderId,
                ],
            ],
            $context
        );

        return $mediaId;
    }


}