<?php declare(strict_types=1);

namespace VideoAISoftware\Content\ProductAiVideo;


use Doctrine\DBAL\ArrayParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Shopware\Core\Content\Product\Aggregate\ProductMedia\ProductMediaDefinition;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Event\EntityDeleteEvent;
use Shopware\Core\Framework\Uuid\Uuid;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\Attribute\AsEventListener;
use VideoAISoftware\Util\ContextHelper;

#[AsEventListener]
readonly class ProductAiVideoUpdater
{
    public function __construct(
        #[Autowire(service: ProductAiVideoDefinition::ENTITY_NAME . '.repository')]
        private EntityRepository $aiVideoRepository,
        private Connection       $connection,
        private ContextHelper    $contextHelper
    )
    {
    }


    public function __invoke(EntityDeleteEvent $event): void
    {
        $ids = $event->getIds(ProductMediaDefinition::ENTITY_NAME);

        if (count($ids) < 1) return;

        $productAiVideoIds = $this->fetchProductAiVideoIdsForProductMediaIds($ids);

        $event->addSuccess(function () use ($productAiVideoIds) {
            $context = $this->contextHelper->createCliContext();
            $idsToDelete = $this->fetchProductAiVideoIdsToDelete($productAiVideoIds);
            $idsToUpdate = array_filter($productAiVideoIds, fn (string $id) => !in_array($id, $idsToDelete));

            if (!empty($idsToDelete)) {
                $this->aiVideoRepository->delete(array_map(fn(string $id) => ['id' => $id], $idsToDelete), $context);
            }

            if (!empty($idsToUpdate)) {
                $this->aiVideoRepository->update(array_map(fn(string $id) => [
                    'id' => $id,
                    'published' => false
                ], $idsToUpdate), $this->contextHelper->createCliContext());
            }
        });
    }

    private function fetchProductAiVideoIdsToDelete(array $ids)
    {
        return $this->connection->createQueryBuilder()
            ->select('lower(hex(id)) as id')
            ->from(ProductAiVideoDefinition::ENTITY_NAME)
            ->where('id in (:ids)')
            ->andWhere('parent_id is not null')
            ->setParameter('ids', Uuid::fromHexToBytesList($ids), ArrayParameterType::BINARY)
            ->executeQuery()
            ->fetchFirstColumn();
    }

    /**
     * @returns string[]
     * @throws Exception
     */
    private function fetchProductAiVideoIdsForProductMediaIds(array $productMediaIds): array
    {
        return $this->connection->createQueryBuilder()
            ->select('lower(hex(id)) as id')
            ->from(ProductAiVideoDefinition::ENTITY_NAME)
            ->where('product_media_id in (:productMediaId)')
            ->setParameter('productMediaId', Uuid::fromHexToBytesList($productMediaIds), ArrayParameterType::BINARY)
            ->executeQuery()
            ->fetchFirstColumn();
    }
}