<?php declare(strict_types=1);

namespace VideoAISoftware\Util;

use Doctrine\DBAL\Connection;
use Shopware\Core\Defaults;
use Shopware\Core\Framework\Api\Context\ContextSource;
use Shopware\Core\Framework\Api\Context\SystemSource;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\Context\BaseContextFactory;
use Shopware\Core\System\SalesChannel\SalesChannelException;

readonly class ContextHelper
{
    public function __construct(
        private ShopwareVersionHelper $shopwareVersionHelper,
        private Connection $connection
    )
    {
    }

    /**
     * Creates a new context by calling Context::createCliContext or Context::createDefaultContext, depending on the currently installed Shopware version.
     *
     * @param ContextSource|null $source
     * @return Context
     */
    public function createCliContext(?ContextSource $source = null): Context
    {
        if (version_compare(
            $this->shopwareVersionHelper->getCurrentVersion(),
            "6.6.1.0",
            "<"
        )) {
            return Context::createDefaultContext($source);
        }

        /** @noinspection PhpUndefinedMethodInspection */
        return Context::createCliContext($source);
    }

    public function createContextForLanguage(string $languageId): Context
    {
        return new Context(
            source: new SystemSource(),
            languageIdChain: $this->buildLanguageChain($languageId)  /** @see BaseContextFactory::buildLanguageChain() */
        );
    }

    /**
     * @return string[]
     */
    private function buildLanguageChain(string $languageId): array
    {
        if ($languageId === Defaults::LANGUAGE_SYSTEM) {
            return [Defaults::LANGUAGE_SYSTEM];
        }

        return array_filter([$languageId, $this->getParentLanguageId($languageId), Defaults::LANGUAGE_SYSTEM]);
    }

    private function getParentLanguageId(string $languageId): ?string
    {
        $data = $this->connection->createQueryBuilder()
            ->select('LOWER(HEX(language.parent_id))')
            ->from('language')
            ->where('language.id = :id')
            ->setParameter('id', Uuid::fromHexToBytes($languageId))
            ->executeQuery()
            ->fetchOne();

        if ($data === false) {
            throw SalesChannelException::languageNotFound($languageId);
        }

        return $data;
    }
}