<?php declare(strict_types=1);

namespace VideoAISoftware\Util;

use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\Language\LanguageEntity;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Symfony\Contracts\Service\ResetInterface;

class PluginConfigService implements ResetInterface
{
    private ?string $productLanguageIsoCode = null;

    public function __construct(
        private readonly SystemConfigService $systemConfigService,
        private readonly EntityRepository $languageRepository,
        private readonly ContextHelper $contextHelper
    )
    {
    }

    public function getApiHost() {
        return $this->systemConfigService->getString('VideoAISoftware.config.apiHost');
    }

    public function getApiKey() {
        return $this->systemConfigService->getString('VideoAISoftware.config.apiKey');
    }

    public function useProductProperties(): bool
    {
        return $this->systemConfigService->get('VideoAISoftware.config.useProductProperties') === true;
    }

    public function useManufacturerInformation(): bool
    {
        return $this->systemConfigService->get('VideoAISoftware.config.useManufacturerInformation') === true;
    }

    public function useProductPrice(): bool
    {
        return $this->systemConfigService->get('VideoAISoftware.config.useProductPrice') === true;
    }

    public function useProductImages(): bool
    {
        return $this->systemConfigService->get('VideoAISoftware.config.useProductImages') === true;
    }

    public function getVideoTemplateId(): ?string
    {
        return $this->systemConfigService->get('VideoAISoftware.config.videoTemplateId');
    }

    public function getBrandName(): ?string
    {
        return $this->systemConfigService->get('VideoAISoftware.config.brandName');
    }

    public function getBrandDescription(): ?string
    {
        return $this->systemConfigService->get('VideoAISoftware.config.brandDescription');
    }

    public function getBrandTargetAudiences(): ?string
    {
        return $this->systemConfigService->get('VideoAISoftware.config.brandTargetAudiences');
    }

    public function getFormOfAddress(): ?string
    {
        return $this->systemConfigService->get('VideoAISoftware.config.formOfAddress');
    }

    public function getProductLanguageId(): ?string
    {
        return $this->systemConfigService->get('VideoAISoftware.config.productLanguageId');
    }

    public function getProductLanguageIsoCode(): ?string
    {
        if (!$this->getProductLanguageId()) return null;

        if (!$this->productLanguageIsoCode) {
            /** @var ?LanguageEntity $language */
            $language = $this->languageRepository->search(
                (new Criteria([$this->getProductLanguageId()]))
                    ->addAssociation('locale'),
                $this->contextHelper->createCliContext()
            )->first();

            if (!$language) return null;

            $this->productLanguageIsoCode = $language->getLocale()->getCode();
        }

        return $this->productLanguageIsoCode;
    }

    public function reset(): void
    {
        $this->productLanguageIsoCode = null;
    }
}