import template from './mwsvais-usage-statistics.html.twig';

const { Component } = Shopware;

Component.register('mwsvais-usage-statistics', {
    template,
    inject: ['fetchUsageStatistics'],

    data() {
        return {
            isLoading: true,
            statistics: [],
            columns: [
                {
                    property: 'month',
                    label: this.$tc('mwsvais-usage-statistics.columns.month'),
                    primary: true,
                    sortable: true,
                },
                {
                    property: 'year',
                    label: this.$tc('mwsvais-usage-statistics.columns.year'),
                    sortable: true,
                },
                {
                    property: 'generated-minutes',
                    label: this.$tc('mwsvais-usage-statistics.columns.generated-minutes'),
                    sortable: true,
                },
            ],
            sortBy: 'month',
            sortDirection: 'DESC',
            currentPage: 1,
            itemsPerPage: 10,
        };
    },

    created() {
        this.loadUsageStatistics();
    },

    methods: {

        onPageSizeChange(event) {
            this.itemsPerPage = event.limit
            this.currentPage = 1;
        },

        loadUsageStatistics() {
            this.isLoading = true;
            this.fetchUsageStatistics.getUsageStatistics({ apiKey: this.apiKey })
                .then(response => {
                    this.statistics = this.formatStatistics(response);
                })
                .catch(error => {
                    console.error('Failed to load usage statistics:', error);
                    this.statistics = [];
                })
                .finally(() => {
                    this.isLoading = false;
                });
        },

        formatStatistics(response) {
            if (!response || !response.statistics) {
                return [];
            }

            return response.statistics.map(stat => ({
                month: stat.month,
                year: stat.year,
                'generated-minutes': stat['videoDuration'],
            }));
        },

        onSortColumn(column) {
            if (this.sortBy === column.property) {
                this.sortDirection = this.sortDirection === 'ASC' ? 'DESC' : 'ASC';
            } else {
                this.sortBy = column.property;
                this.sortDirection = 'ASC';
            }
            this.paginatedStatistics;
        },

        onPageChange(page) {
            this.currentPage = page.page;
            this.itemsPerPage = page.limit;
        },

        sortStatistics(stats) {
            return stats.sort((a, b) => {
                const aValue = a[this.sortBy];
                const bValue = b[this.sortBy];
                if (aValue < bValue) return this.sortDirection === 'ASC' ? -1 : 1;
                if (aValue > bValue) return this.sortDirection === 'ASC' ? 1 : -1;
                return 0;
            });
        },
    },

    computed: {
        apiKey() {
            return this.pluginConfig ? this.pluginConfig['VideoAISoftware.config.apiKey'] : '';
        },
        pluginConfig() {
            let $parent = this.$parent;

            while ($parent.actualConfigData === undefined) {
                $parent = $parent.$parent;
            }

            return $parent.actualConfigData.null;
        },
        paginatedStatistics() {
            if (!this.statistics || this.statistics.length === 0) return [];
            const sortedStats = this.sortStatistics([...this.statistics]);
            const start = (this.currentPage - 1) * this.itemsPerPage;
            const end = start + this.itemsPerPage;
            return sortedStats.slice(start, end);
        },
        totalStatistics() {
            return this.statistics.length;
        },
    }
});
