<?php declare(strict_types=1);

namespace VideoAISoftware\Controller\Api\PluginUpdate;

use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\Api\Controller\ApiController;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Attribute\MapRequestPayload;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Contracts\Service\Attribute\Required;
use VideoAISoftware\Component\Notification\NotificationService;
use VideoAISoftware\Util\PluginConfigService;
use VideoAISoftware\Util\ValidateWebhookSignature;

class PluginUpdateWebhookController extends ApiController
{
    private readonly PluginConfigService $pluginConfigService;
    private readonly NotificationService $notificationService;
    private readonly LoggerInterface $logger;

    #[Required]
    public function setPluginConfigService(PluginConfigService $pluginConfigService): void
    {
        $this->pluginConfigService = $pluginConfigService;
    }

    #[Required]
    public function setNotificationService(NotificationService $notificationService): void
    {
        $this->notificationService = $notificationService;
    }

    #[Required]
    public function setLogger(#[Autowire(service: 'monolog.logger.business_events')] LoggerInterface $logger): void
    {
        $this->logger = $logger;
    }

    #[Route(path: '/mws-vais/plugin-update/webhook', name: 'frontend.mws_vais.plugin.update.webhook', defaults: ['_routeScope' => ['storefront']], methods: ['POST'])]
    public function handleWebhook(
        #[MapRequestPayload] PluginUpdateWebhookDto $dto,
        Request                                     $rawRequest
    ) {
        try {
            $signatureFromRequestHeader = $rawRequest->headers->get('Signature');
            if (!$signatureFromRequestHeader) {
                throw new AccessDeniedHttpException('Missing webhook signature');
            }
            $apiToken = $this->pluginConfigService->getApiKey();
            ValidateWebhookSignature::validate($signatureFromRequestHeader, $rawRequest->getContent(), $apiToken);

            $this->notificationService->createNotification(
                'mws_vais.plugin.update',
                [
                    'version' => $dto->version,
                    'url' => $dto->url
                ]
            );

            return new JsonResponse();
        } catch (\Throwable $throwable) {
            $this->logger->error("Video-Script-Webhook failed: " . $throwable->getMessage(), [
                'request' => [
                    'version' => $dto->version,
                    'url' => $dto->url
                ],
                'exception' => [
                    'file' => $throwable->getFile(),
                    'line' => $throwable->getLine(),
                    'exception-class' => get_class($throwable),
                    'trace' => $throwable->getTrace(),
                ]
            ]);
            throw $throwable;
        }
    }
}