import template from './sw-product-list.html.twig';

const {Criteria} = Shopware.Data;

Shopware.Component.override('sw-product-list', {
    template,

    inject: [
        'repositoryFactory',
        'MwsVaisAiVideoManager', 
        'videoScriptControl'
    ],

    data() {
      return {
          isAiBulkEdit: false,
      }
    },

    computed: {
        anyProductsSelected() {
            return this.selectionCount > 0;
        },

        aiVideoRepository() {
            return this.repositoryFactory.create('mws_vais_product_ai_video');
        },
    },

    methods: {
        async generateVideoScripts(scriptAndVideo) {
            this.isLoading = true;

            try {
                const productIds = this.selectionArray.map((product) => product.id);

                const response = await this.videoScriptControl.generateBulk(productIds, scriptAndVideo)

                if (response.errorCount === 0) {
                    this.createNotificationSuccess({
                        title: this.$tc('mwsvais.product-list.button.generate-video-scripts.label'),
                        message: this.$tc('mwsvais.product-list.button.generate-video-scripts.success'),
                    });
                } else {
                    this.createNotificationWarning({
                        title: this.$tc('mwsvais.product-list.button.generate-video-scripts.label'),
                        message: this.$tc('mwsvais.product-list.button.generate-video-scripts.error', {
                            count: response.count,
                            total: response.total
                        }),
                    })
                }

            } catch(error) {
                this.createNotificationError({
                    title: this.$tc('mwsvais.product-list.button.generate-video-scripts.label'),
                    message: error.message ? this.$tc(error.message) : error
                });
            }

            this.isLoading = false;
        },

        async generateVideos() {
            this.isLoading = true;
            
            try {
                const productIds = this.selectionArray.map((product) => product.id);

                const response = await this.MwsVaisAiVideoManager.generateBulk(productIds);

                if (response.errorCount === 0) {
                    this.createNotificationSuccess({
                        title: this.$tc('mwsvais.product-list.button.generate-videos.label'),
                        message: this.$tc('mwsvais.product-list.button.generate-videos.success'),
                    });
                } else {
                    this.createNotificationWarning({
                        title: this.$tc('mwsvais.product-list.button.generate-videos.label'),
                        message: this.$tc('mwsvais.product-list.button.generate-videos.error', {
                            count: response.count,
                            total: response.total
                        }),
                    })
                }

            } catch(error) {
                this.createNotificationError({
                    title: this.$tc('mwsvais.product-list.button.generate-videos.label'),
                    message: error.message ? this.$tc(error.message) : error
                });
            }

            this.isLoading = false;
        },

        async publishVideos() {
            this.isLoading = true;

            try {
                const productIds = this.selectionArray.map((product) => product.id);

                for(const productId of productIds) {
                    const aiVideos = await this.fetchAiVideos(productId);
                    await this.publishAIVideos(aiVideos);
                }

                this.createNotificationSuccess({
                    title: this.$tc('mwsvais.product-list.button.publish-videos.label'),
                    message: this.$tc('mwsvais.product-list.button.publish-videos.success'),
                });
            } catch(error) {
                this.createNotificationError({
                    title: this.$tc('mwsvais.product-list.button.publish-videos.label'),
                    message: error.message ? this.$tc(error.message) : error
                });
            }

            this.isLoading = false;
        },

        async fetchAiVideos(productId) {
            const criteria = new Criteria();
            criteria.addFilter(
                Criteria.equals('productId', productId)
            );
            criteria.addFilter(
                Criteria.equals('published', false)
            );

            return await this.aiVideoRepository.search(criteria, Shopware.Context.Api)
        },

        async publishAIVideos(aiVideos) {
            for(const aiVideo of aiVideos) {
                await this.MwsVaisAiVideoManager.publishVideo(aiVideo.productId, aiVideo.mediaId);
            }
        },

        async showAiBulkEditItemsModal() {
            this.isAiBulkEdit = true;
            this.showBulkEditModal = true;
        },

        onBulkEditItems() {
            if (this.isAiBulkEdit === false) {
                this.$super('onBulkEditItems');
                return;
            }
            this.isAiBulkEdit = false;
            this.$router.push({
                name: 'mwsvais.ai.bulk.edit.product',
            });
        },
    },
});