// eslint-disable-next-line plugin-rules/no-src-imports
import {NotificationType} from "@administration/app/mixin/notification.mixin"
// eslint-disable-next-line plugin-rules/no-src-imports
import {FullState} from "@administration/core/factory/state.factory"
// eslint-disable-next-line plugin-rules/no-src-imports
import {RouteLocationNormalized} from "@administration/../node_modules/vue-router/"
import emitter from '@MwsVais/util/mitt'

const {Application} = Shopware

interface NotificationItemAction {
    label: string
    route: string | Partial<RouteLocationNormalized>
}

interface NotificationPayload {
    timestamp: string
}

interface NotificationItem {
    title: string
    message: string
    variant: NotificationType
    actions: NotificationItemAction[]
    timestamp: string
    growl?: boolean
}

interface WebhookNotificationHandler {
    key: string
    forceNotificationCenter?: true
    notificationCenterVariant?: Exclude<NotificationType, "success">
    handle(notificationPayload: NotificationPayload): Omit<NotificationItem, "timestamp">|false
}

/**
 * This service listens to the notification/createNotification action and overrides the notification.
 * Shopware does not allow database notifications to define title or actions.
 * Misusing notification->status to define a key used by the formatters.
 * Misusing notification->message to store the payload.
 * Handlers are used to override the notification based on the key, replacing message and variant (=status) and adding title/actions.
 *
 */
export default class NotificationWebhookService {
    private readonly subscribedActions = ['notification/createNotification']
    private readonly handlers: WebhookNotificationHandler[] = []

    constructor() {
        this.handlers.push(
            VideoScriptWebhookHandler,
            GenerateVideoWebhookHandler,
            PluginUpdateWebhookHandler,
        )
    }

    init() {
            /*
             * (Shopware.State as FullState) is somehow only needed for vscode?
             * Shopware.State in phpstorm already infers the type correctly, but vscode doesn't?
             * (I) don't know why ... (♪♪ https://youtu.be/tO4dxvguQDk ♪♪)
             */
            (Shopware.State as FullState).subscribeAction({
            before: (action) => {
                if (!this.subscribedActions.includes(action.type)) {
                    return
                }

                // Execute before the action is executed -> modify the payload before it is committed
                const notification = action.payload as NotificationItem

                const handler = this.handlers.find((formatter) => formatter.key === notification.variant)

                if (!handler) {
                    return
                }

                try {
                    const notificationPayload = JSON.parse(notification.message) as NotificationPayload
                    const result = handler.handle(notificationPayload)
                    
                    if (result === false) {
                        notification.title = undefined
                        notification.message = undefined
                        notification.growl = undefined
                        notification.variant = undefined
                        notification.actions = undefined
                    } else {
                        Object.assign(notification, result)
                    }

                    // Shopware won't add success notifications to notification center ...
                    // Dispatch a second notification to add success notification to notification center
                    // Must be explicitly enabled in handler
                    if (handler.forceNotificationCenter === true
                        && notification.variant === 'success'
                    ) {
                        (Shopware.State as FullState).dispatch('notification/createNotification', {
                            ...notification,
                            variant: handler.notificationCenterVariant ?? 'info',
                            growl: false
                        } as NotificationItem)
                    }
                } catch (e) {
                    // JSON.parse(notification.message) could throw an error
                    // if the message is not a valid JSON string
                    // in this case we just ignore the error
                }
            }
        })
    }
}

const VideoScriptWebhookHandler: WebhookNotificationHandler = {
    key: 'mws_vais.video_script.webhook',
    forceNotificationCenter: true,
    handle(notificationPayload: NotificationPayload & {
        productId: string,
        success: boolean,
        bulkData: {
            finished: number,
            total: number
        } | null
    }) {
        if (!notificationPayload.success) {

            emitter.emit('mws_vais.video_script.webhook', {
                productId: notificationPayload.productId,
                success: false
            })
            return {
                title: Shopware.Snippet.tc('mwsvais.product-detail.field.video-script-button.error.title'),
                message: Shopware.Snippet.tc('mwsvais.product-detail.field.video-script-button.error.message'),
                variant: 'error',
                actions: [
                    {
                        label: Shopware.Snippet.tc('global.entities.product'),
                        route: {
                            name: 'sw.product.detail.ai.video',
                            params: {
                                id: notificationPayload.productId
                            }
                        }
                    }
                ]
            };
        }

        emitter.emit('mws_vais.video_script.webhook', {
            productId: notificationPayload.productId,
            success: true
        })

        const message = notificationPayload.bulkData
            ? Shopware.Snippet.tc('mwsvais.product-detail.field.video-script-button.successBulk', {
                finished: notificationPayload.bulkData.finished,
                total: notificationPayload.bulkData.total
            })
            : Shopware.Snippet.tc('mwsvais.product-detail.field.video-script-button.success')

        return {
            title: Shopware.Snippet.tc('mwsvais.product-detail.field.video-script-button.label'),
            message,
            variant: 'success',
            actions: [
                {
                    label: Shopware.Snippet.tc('global.entities.product'),
                    route: {
                        name: 'sw.product.detail.ai.video',
                        params: {
                            id: notificationPayload.productId
                        }
                    }
                }
            ]
        }
    }
}

const GenerateVideoWebhookHandler : WebhookNotificationHandler = {
    key: 'mws_vais.generate_video.webhook',
    forceNotificationCenter: true,
    handle(notificationPayload: NotificationPayload & {
        productId: string,
        success: boolean,
        bulkData: {
            finished: number,
            total: number
        } | null
    }) {
        if (!notificationPayload.success) {
            emitter.emit('mws_vais.generate_video.webhook', {
                productId: notificationPayload.productId,
                success: false
            })

            return {
                title: Shopware.Snippet.tc('mwsvais.product-detail.video-control.generateVideoNotification.error.title'),
                message: Shopware.Snippet.tc('mwsvais.product-detail.video-control.generateVideoNotification.error.message'),
                variant: 'error',
                actions: [
                    {
                        label: Shopware.Snippet.tc('global.entities.product'),
                        route: {
                            name: 'sw.product.detail.ai.video',
                            params: {
                                id: notificationPayload.productId
                            }
                        }
                    }
                ]
            };
        }

        emitter.emit('mws_vais.generate_video.webhook', {
            productId: notificationPayload.productId,
            success: true
        })

        const message = notificationPayload.bulkData
            ? Shopware.Snippet.tc('mwsvais.product-detail.video-control.generateVideoNotification.successBulk.message', {
                finished: notificationPayload.bulkData.finished,
                total: notificationPayload.bulkData.total
            })
            : Shopware.Snippet.tc('mwsvais.product-detail.video-control.generateVideoNotification.success.message')

        return {
            title: Shopware.Snippet.tc('mwsvais.product-detail.video-control.generateVideoNotification.success.title'),
            message: message,
            variant: 'success',
            actions: [
                {
                    label: Shopware.Snippet.tc('global.entities.product'),
                    route: {
                        name: 'sw.product.detail.ai.video',
                        params: {
                            id: notificationPayload.productId
                        }
                    }
                }
            ]
        }
    }
}

const PluginUpdateWebhookHandler : WebhookNotificationHandler = {
    key: 'mws_vais.plugin.update',
    forceNotificationCenter: true,
    notificationCenterVariant: 'info',
    handle(notificationPayload: NotificationPayload & {
        version: string
        url: string
    }): Omit<NotificationItem, "timestamp"> | false {
        const applicationRoot = Application.getApplicationRoot();


        return {
            title: 'Video AI Software',
            message: applicationRoot.$tc('mwsvais.version-update-webhook.message', {
                version: notificationPayload.version
            }),
            variant: 'info',
            actions: [{
                label: applicationRoot.$tc('mwsvais.version-update-webhook.download-label'),
                route: notificationPayload.url
            }]
        }
    }
}