<?php declare(strict_types=1);

namespace VideoAISoftware\Controller\Api\GenerateVideoScriptController;

use Psr\Log\LoggerInterface;
use Shopware\Core\Content\Media\Core\Application\AbstractMediaUrlGenerator;
use Shopware\Core\Content\Media\Core\Params\UrlParams;
use Shopware\Core\Content\Product\Aggregate\ProductMedia\ProductMediaEntity;
use Shopware\Core\Content\Product\Exception\ProductNotFoundException;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Framework\Api\Controller\ApiController;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\NotFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\PrefixFilter;
use Shopware\Core\Framework\Uuid\Uuid;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpKernel\Attribute\MapRequestPayload;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\DecodingExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\Service\Attribute\Required;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Throwable;
use VideoAISoftware\Component\Api\MiddlewareClientFactory;
use VideoAISoftware\Component\Notification\NotificationService;
use VideoAISoftware\Component\ProductAiVideo\VideoManager;
use VideoAISoftware\Content\BulkMeta\BulkMetaDefinition;
use VideoAISoftware\Content\BulkMeta\BulkMetaEntity;
use VideoAISoftware\Content\BulkMeta\BulkState;
use VideoAISoftware\Content\ProductSetting\GenerateVideoState;
use VideoAISoftware\Content\ProductSetting\Media\ProductSettingMediaCollection;
use VideoAISoftware\Content\ProductSetting\Media\ProductSettingMediaDefinition;
use VideoAISoftware\Content\ProductSetting\Media\ProductSettingMediaEntity;
use VideoAISoftware\Content\ProductSetting\ProductSettingEntity;
use VideoAISoftware\Content\ProductSetting\ProductSettingDefinition;
use VideoAISoftware\Content\ProductSetting\VideoScriptState;
use VideoAISoftware\Util\ContextHelper;
use VideoAISoftware\Util\PluginConfigService;
use VideoAISoftware\Util\ValidateWebhookSignature;
#[Route(defaults: ['_routeScope' => ['api']])]
class GenerateVideoScriptController extends ApiController
{

    private readonly EntityRepository $productRepository;

    private readonly EntityRepository $currencyRepository;
    private readonly EntityRepository $productSettingRepository;
    private readonly EntityRepository $bulkMetaRepository;
    private readonly EntityRepository $productMediaRepository;
    private readonly EntityRepository $productSettingMediaRepository;

    private readonly MiddlewareClientFactory $clientFactory;

    private readonly PluginConfigService $pluginConfigService;

    private readonly NotificationService $notificationService;

    private readonly ContextHelper $contextHelper;

    private readonly VideoManager $videoManager;

    private readonly AbstractMediaUrlGenerator $mediaUrlGenerator;

    private readonly LoggerInterface $logger;

    #[Required]
    public function setProductRepository(EntityRepository $productRepository): void
    {
        $this->productRepository = $productRepository;
    }

    #[Required]
    public function setCurrencyRepository(EntityRepository $currencyRepository): void
    {
        $this->currencyRepository = $currencyRepository;
    }

    #[Required]
    public function setProductSettingRepository(
        #[Autowire(service: ProductSettingDefinition::ENTITY_NAME . '.repository')]
        EntityRepository $productSettingRepository
    ): void
    {
        $this->productSettingRepository = $productSettingRepository;
    }

    #[Required]
    public function setProductSettingMediaRepository(
        #[Autowire(service: ProductSettingMediaDefinition::ENTITY_NAME . '.repository')]
        EntityRepository $productSettingMediaRepository
    ): void
    {
        $this->productSettingMediaRepository = $productSettingMediaRepository;
    }

    #[Required]
    public function setClientFactory(MiddlewareClientFactory $clientFactory): void
    {
        $this->clientFactory = $clientFactory;
    }

    #[Required]
    public function setPluginConfigService(PluginConfigService $pluginConfigService): void
    {
        $this->pluginConfigService = $pluginConfigService;
    }

    #[Required]
    public function setNotificationService(NotificationService $notificationService): void
    {
        $this->notificationService = $notificationService;
    }

    #[Required]
    public function setContextHelper(ContextHelper $contextHelper): void
    {
        $this->contextHelper = $contextHelper;
    }

    #[Required]
    public function setBulkMetaRepository(
        #[Autowire(service: BulkMetaDefinition::ENTITY_NAME . '.repository')]
        EntityRepository $bulkMetaRepository): void
    {
        $this->bulkMetaRepository = $bulkMetaRepository;
    }

    #[Required]
    public function setProductMediaRepository(EntityRepository $productMediaRepository): void
    {
        $this->productMediaRepository = $productMediaRepository;
    }

    #[Required]
    public function setVideoManager(VideoManager $videoManager): void
    {
        $this->videoManager = $videoManager;
    }

    #[Required]
    public function setMediaUrlGenerator(AbstractMediaUrlGenerator $mediaUrlGenerator): void
    {
        $this->mediaUrlGenerator = $mediaUrlGenerator;
    }

    #[Required]
    public function setLogger(#[Autowire(service: 'monolog.logger.business_events')] LoggerInterface $logger): void
    {
        $this->logger = $logger;
    }

    #[Route(path: '/api/video-script/generate-bulk', name: 'api.action.video-script.generate.bulk', methods: ['POST'])]
    public function generateBulk(
        #[MapRequestPayload] VideoScriptBulkDto $request
    ): JsonResponse
    {
        $context = $this->contextHelper->createContextForLanguage(
            $this->pluginConfigService->getProductLanguageId()
        );

        $bulkId = Uuid::randomHex();

        $this->bulkMetaRepository->create(array_map(fn(string $productId) => [
            'bulkId' => $bulkId,
            'productId' => $productId,
            'bulkState' => BulkState::PENDING->value,
            'scriptAndVideo' => $request->scriptAndVideo,
            'context' => serialize($context)
        ], $request->productIds), $context);

        $errors = [];

        foreach ($request->productIds as $productId) {
            try {
                $this->handleProduct($productId, $context, $request->scriptAndVideo);
                continue;
            } catch (ClientExceptionInterface|ServerExceptionInterface $exception) {
                $this->handleHttpException($productId, $exception, $context);
            } catch (Throwable $exception) {
                $this->handleException($productId, $exception, $context);
            }

            $errors[] = [
                'bulkId' => $bulkId,
                'productId' => $productId,
                'bulkState' => VideoScriptState::ERROR->value
            ];
        }

        if (!empty($errors)) {
            $this->bulkMetaRepository->update($errors, $context);
        }

        return new JsonResponse([
            'count' => count($request->productIds) - count($errors),
            'errorCount' => count($errors),
            'total' => count($request->productIds)
        ]);
    }

    #[Route(path: '/api/video-script/generate/{productId}', name: 'api.action.video-script.generate', methods: ['POST'])]
    public function generate(
        string $productId
    ): JsonResponse
    {
        $context = $this->contextHelper->createContextForLanguage(
            $this->pluginConfigService->getProductLanguageId()
        );

        try {
            $middlewareMessage = $this->handleProduct(
                $productId,
                $context,
                false
            );

            return new JsonResponse($middlewareMessage);
        } catch (ClientExceptionInterface|ServerExceptionInterface $e) {
            $errorResponse = $this->handleHttpException($productId, $e, $context);
            return new JsonResponse($errorResponse, $e->getResponse()->getStatusCode());
        } catch (Throwable $exception) {
            $errorResponse = $this->handleException($productId, $exception, $context);
            return new JsonResponse($errorResponse, 500);
        }
    }

    /**
     * @throws TransportExceptionInterface
     * @throws ServerExceptionInterface
     * @throws RedirectionExceptionInterface
     * @throws DecodingExceptionInterface
     * @throws ClientExceptionInterface
     */
    private function handleProduct(
        string  $productId,
        Context $context,
        bool    $scriptAndVideo
    ): array
    {
        $product = $this->fetchProduct($productId, $context);

        /** @var ?ProductSettingEntity $setting */
        $setting = $product->getExtension(ProductSettingDefinition::ENTITY_EXTENSION);

        /** @var ?ProductSettingMediaCollection $selectedProductMedia */
        $selectedProductMedia = $product->getExtension(ProductSettingMediaDefinition::ENTITY_EXTENSION) ?? new ProductSettingMediaCollection();

        if ($setting?->getVideoScriptState() === VideoScriptState::PENDING) {
            throw new \Exception('Already in progress');
        }

        if ($scriptAndVideo && $setting?->getGenerateVideoState() === GenerateVideoState::PENDING) {
            throw new \Exception('Already in progress');
        }

        $useProperties = $setting?->getUseProductProperties() ?? $this->pluginConfigService->useProductProperties();
        $useManufacturerInformation = $setting?->getUseManufacturerInformation() ?? $this->pluginConfigService->useManufacturerInformation();
        $useProductPrice = $setting?->getUseProductPrice() ?? $this->pluginConfigService->useProductPrice();
        $useProductImages = $setting?->getUseProductImages() ?? $this->pluginConfigService->useProductImages();

        $price = null;
        if ($useProductPrice) {
            $price = $product->getCurrencyPrice($this->fetchCurrencyId($context))->getGross();
        }

        $productMediaUrls = null;

        if ($useProductImages) {
            if ($selectedProductMedia->count() < 1) {
                // Selected product media not initialized, e.g. product AI tab page never has been opened
                $selectedProductMedia = $this->initProductMedia($product->getId(), $context);
            }

            if ($selectedProductMedia->count() > 0) {
                $productMediaUrls = array_values($selectedProductMedia->map(function (ProductSettingMediaEntity $entity) {
                    $paths = $this->mediaUrlGenerator->generate(
                        [UrlParams::fromMedia($entity->getProductMedia()->getMedia())]
                    );
                    return array_shift($paths);
                }));
            }
        }

        $this->productSettingRepository->upsert([[
            'productId' => $productId,
            'videoScriptState' => VideoScriptState::PENDING->value,
        ]], $context);


        $response = $this->clientFactory->create()->request('POST', 'video-scripts', [
            'body' => [
                'outputLanguage' => $this->pluginConfigService->getProductLanguageIsoCode(),
                'id' => $productId,
                'formOfAddress' => $setting?->getFormOfAddress() ?? $this->pluginConfigService->getFormOfAddress(),
                'brandName' => $this->pluginConfigService->getBrandName(),
                'brandDescription' => $this->pluginConfigService->getBrandDescription(),
                'targetAudience' => $this->pluginConfigService->getBrandTargetAudiences(),
                'templateId' => $setting?->getVideoTemplateId() ?? $this->pluginConfigService->getVideoTemplateId(),
                'productName' => $product->getTranslation('name'),
                'productDescription' => $product->getTranslation('description'),
                'productProperties' => $useProperties ? $this->formatProperties($product) : null,
                'manufacturerInformation' => $useManufacturerInformation ? $product->getManufacturer()?->getTranslation('description') : null,
                'additionalImportantInformation' => $setting?->getAdditionalVideoContent(),
                'informationToExclude' => $setting?->getExcludedContent(),
                'additionalInstructionsForYou' => $setting?->getAdditionalInstructions(),
                'productPrice' => $price,
                'productImages' => $productMediaUrls
            ]
        ]);

        $json = $response->toArray();

        return [
            'message' => $json['message']
        ];
    }

    private function handleHttpException(
        string                                            $productId,
        ClientExceptionInterface|ServerExceptionInterface $exception,
        Context                                           $context
    ): array
    {
        $responseData = $exception->getResponse()->toArray(false);

        $this->productSettingRepository->upsert([[
            'productId' => $productId,
            'videoScriptState' => VideoScriptState::ERROR->value,
            'videoScriptError' => $responseData
        ]], $context);

        return $responseData;
    }

    private function handleException(
        string    $productId,
        Throwable $throwable,
        Context   $context
    ): array
    {
        $responseData = [
            'message' => $throwable->getMessage(),
            'trace' => $throwable->getTrace()
        ];

        $this->productSettingRepository->upsert([[
            'productId' => $productId,
            'videoScriptState' => VideoScriptState::ERROR->value,
            'videoScriptError' => $responseData
        ]], $context);

        return $responseData;
    }


    #[Route(path: '/mws-vais/video-script/webhook', name: 'frontend.mws_vais.action.video_script.webhook', defaults: ['_routeScope' => ['storefront']], methods: ['POST'])]
    public function webhook(
        #[MapRequestPayload] VideoScriptWebhookDto $request,
        Request $rawRequest
    ): JsonResponse
    {
        try {
            $signatureFromRequestHeader = $rawRequest->headers->get('Signature');
            if (!$signatureFromRequestHeader) {
                throw new AccessDeniedHttpException('Missing webhook signature');
            }
            $apiToken = $this->pluginConfigService->getApiKey();
            ValidateWebhookSignature::validate($signatureFromRequestHeader, $rawRequest->getContent(), $apiToken);

            if ($request->success && !$request->videoScript) {
                return new JsonResponse([
                    'message' => 'VideoScript must not be null when success is true!'
                ], 422);
            }

            $this->validateProductExistence($request->id);

            $context = $this->contextHelper->createCLIContext();

            $videoScript = $request->videoScript;

            if ($videoScript) {
                $this->prepareVideoScript($request->id, $videoScript, $context);
                $videoScript['meta']['templateId'] = $request->templateId;
            }

            $this->productSettingRepository->upsert([[
                'productId' => $request->id,
                'videoScriptState' => $request->success ? VideoScriptState::SUCCESS->value : VideoScriptState::ERROR->value,
                'videoScript' => $videoScript,
                'videoScriptError' => $request->error,
            ]], $context);

            $bulkData = $this->handleBulk($request);

            // bulkData is false if notification should be skipped
            if ($bulkData !== false) {
                $this->notificationService->createNotification(
                    'mws_vais.video_script.webhook',
                    [
                        'productId' => $request->id,
                        'success' => $request->success,
                        'bulkData' => $bulkData
                    ]
                );
            }

            return new JsonResponse(['message' => 'Successfully received']);
        } catch (Throwable $throwable) {
            $this->logger->error("Video-Script-Webhook failed: " . $throwable->getMessage(), [
                'request' => [
                    'id' => $request->id,
                    'success' => $request->success,
                    'video-script' => $request->videoScript,
                    'template-id' => $request->templateId,
                    'error' => $request->error,
                ],
                'exception' => [
                    'file' => $throwable->getFile(),
                    'line' => $throwable->getLine(),
                    'exception-class' => get_class($throwable),
                    'trace' => $throwable->getTrace(),
                ]
            ]);
            throw $throwable;
        }
    }

    private function prepareVideoScript(string $productId, array &$videoScript, Context $context): void
    {
        if (!array_key_exists('scenes', $videoScript)) return;

        foreach ($videoScript['scenes'] as $sceneKey => $scene) {
            if (!array_key_exists('variables', $scene)) continue;

            foreach ($scene['variables'] as $variableKey => $variable) {
                if (!array_key_exists('variableType', $variable)) continue;
                if ($variable['variableType'] !== 'image') continue;

                if (!array_key_exists('variableOutputValue', $variable)) continue;

                $imageUrl = $variable['variableOutputValue'];

                $videoScript['scenes'][$sceneKey]['variables'][$variableKey]['variableOutputValue'] = $this->findProductMediaByUrl($productId, $imageUrl, $context);
            }
        }
    }

    private function findProductMediaByUrl(string $productId, string $imageUrl, Context $context): ?string
    {
        $path = mb_substr(parse_url($imageUrl)['path'], 1);


        $criteria = new Criteria();
        $criteria->addFilter(
            new EqualsFilter('productId', $productId),
            new EqualsFilter('media.path', $path)
        );

        /** @var ?ProductMediaEntity $productMedia */
        $productMedia = $this->productMediaRepository->search($criteria, $context)->first();

        return $productMedia?->getMediaId();
    }

    private function formatProperties(?ProductEntity $product): ?string
    {
        $productProperties = [];

        foreach ($product->getProperties() as $option) {
            if (!array_key_exists($option->getGroupId(), $productProperties)) {
                $productProperties[$option->getGroupId()] = [
                    'name' => $option->getGroup()->getTranslation('name'),
                    'values' => []
                ];
            }

            $productProperties[$option->getGroupId()]['values'][] = $option->getTranslation('name');
        }

        if (empty($productProperties)) {
            return null;
        }

        $productProperties = array_map(fn(array $group) => "{$group['name']}:" . implode(',', $group['values']), $productProperties);
        return implode(';', $productProperties);
    }

    private function fetchProduct(string $productId, Context $context): ProductEntity
    {
        $criteria = new Criteria([$productId]);
        $criteria->addAssociation(ProductSettingDefinition::ENTITY_EXTENSION);
        $criteria->addAssociation(ProductSettingMediaDefinition::ENTITY_EXTENSION . '.productMedia.media');
        $criteria->addAssociation('properties.group');
        $criteria->addAssociation('manufacturer');

        /** @var ?ProductEntity $product */
        $product = $this->productRepository->search($criteria, $context)->first();

        if (!$product) {
            throw new ProductNotFoundException($productId);
        }

        return $product;
    }

    private function fetchCurrencyId(Context $context): string
    {
        return $this->currencyRepository->searchIds(
            (new Criteria())
                ->addFilter(new EqualsFilter('isoCode', 'EUR')),
            $context
        )->firstId();
    }

    private function validateProductExistence(string $id): void
    {
        if (!$this->productRepository->searchIds(
            new Criteria([$id]),
            $this->contextHelper->createCLIContext()
        )->firstId()) {
            throw new ProductNotFoundException($id);
        }
    }

    private function getBulkMeta(string $productId, Context $context): ?BulkMetaEntity
    {
        return $this->bulkMetaRepository->search(
            (new Criteria())
                ->addFilter(
                    new EqualsFilter('productId', $productId),
                    new EqualsFilter('bulkState', BulkState::PENDING->value)
                ),
            $context
        )->first();
    }

    private function handleBulk(VideoScriptWebhookDto $request): array|false|null
    {
        $context = $this->contextHelper->createCliContext();

        $bulkMeta = $this->getBulkMeta($request->id, $context);

        if (!$bulkMeta) {
            return null;
        }

        if ($bulkMeta->isScriptAndVideo()) {
            // Generate video if flag is set, suppress video script notification
            try {

                $this->videoManager->generateVideo(
                    $request->id,
                    $bulkMeta->getContext()
                );
            } catch (Throwable $exception) {
                $this->notificationService->createNotification(
                    'mws_vais.video_script.webhook',
                    [
                        'productId' => $request->id,
                        'success' => false,
                        'message' => $exception->getMessage()
                    ]
                );
            }
            return false;
        }

        $this->bulkMetaRepository->update([[
            'bulkId' => $bulkMeta->getBulkId(),
            'productId' => $bulkMeta->getProductId(),
            'bulkState' => match ($request->success) {
                true => VideoScriptState::SUCCESS->value,
                false => VideoScriptState::ERROR->value
            }
        ]], $context);

        $finished = $this->bulkMetaRepository->searchIds(
            (new Criteria())
                ->addFilter(
                    new EqualsFilter('bulkId', $bulkMeta->getBulkId()),
                    new EqualsFilter('bulkState', BulkState::SUCCESS->value)
                ),
            $context
        )->getTotal();

        $total = $this->bulkMetaRepository->searchIds(
            (new Criteria())
                ->addFilter(
                    new EqualsFilter('bulkId', $bulkMeta->getBulkId()),
                    new NotFilter(NotFilter::CONNECTION_AND, [
                        new EqualsFilter('bulkState', BulkState::ERROR->value)
                    ])
                ), $context
        )->getTotal();

        return [
            'finished' => $finished,
            'total' => $total
        ];
    }

    private function initProductMedia(string $productId, Context $context): ProductSettingMediaCollection
    {
        $criteria = new Criteria([$productId]);
        $criteria->addAssociation('media');
        $criteria->addFilter(
            new PrefixFilter('media.media.mimeType', 'image')
        );

        /** @var ?ProductEntity $product */
        $product = $this->productRepository->search($criteria, $context)->first();

        if (!$product) return new ProductSettingMediaCollection();

        $productMedia = $product->getMedia();

        $this->productRepository->update([[
            'id' => $productId,
            ProductSettingMediaDefinition::ENTITY_EXTENSION => array_values($productMedia->map(fn (ProductMediaEntity $entity) => [
                'productMediaId' => $entity->getId()
            ]))
        ]], $context);

        return $this->productSettingMediaRepository->search(
            (new Criteria())
                ->addFilter(new EqualsFilter('productId', $productId))
                ->addAssociation('productMedia.media'),
            $context
        )->getEntities();
    }
}
