import './mwsvais-video-control.scss'
import template from './mwsvais-video-control.html.twig'
import emitter from "@MwsVais/util/mitt";
import {saveProduct} from "@MwsVais/util/product";
import { formatDuration } from '@MwsVais/util/duration';

const { mapState } = Shopware.Component.getComponentHelper();
const { Mixin } = Shopware;
const {Criteria} = Shopware.Data;
const { mapPropertyErrors } = Shopware.Component.getComponentHelper();

Shopware.Component.register('mwsvais-video-control', {
    template,

    props: ['mediaItems'],

    inject: ['MwsVaisAiVideoManager', 'repositoryFactory', 'fetchVideoTemplates'],

    mixins: [
        Mixin.getByName('notification')
    ],

    data() {
        return {
            isPublishingVideo: false,
            videoTemplates: {},
            loading: true,
            pending: false,
            key: new Date()
        }
    },

    async created() {
        await this.loadVideoTemplates()
        this.loading = false
    },

    mounted() {
        emitter.on('mws_vais.generate_video.webhook', async (payload) => {
            if (payload.productId !== this.product.id) return

            if (payload.success === false) {
                this.product.extensions.mwsVaisProductVideoConfig.generateVideoState = 'error'
                return
            }

            this.aiVideos = await this.fetchAiVideos()
            this.product.extensions.mwsVaisProductVideoConfig.generateVideoState = 'success'
        })
    },

    unmounted() {
        emitter.off('mws_vais.generate_video.webhook')
    },

    computed: {
        aiVideoRepository() {
            return this.repositoryFactory.create('mws_vais_product_ai_video');
        },

        ...mapState('swProductDetail', [
            'product'
        ]),

        ...mapPropertyErrors('mwsVaisProductSetting', [
            'videoScript'
        ]),

        canGenerateVideo() {
            return !this.isGenerating && this.hasVideoScript;
        },

        hasVideoScript() {
            return this.product.extensions.mwsVaisProductVideoConfig.videoScript != null
        },

        isGenerating() {
            return this.product.extensions.mwsVaisProductVideoConfig.generateVideoState === 'pending'
        },

        aiVideos: {
            set(aiVideos) {
                this.product.extensions.mwsVaisProductAiVideo.length = 0
                this.product.extensions.mwsVaisProductAiVideo.add(...aiVideos);
            },
            get() {
                return this.product?.extensions?.mwsVaisProductAiVideo ?? []
            }
        },

        videoScript() {
            return this.product?.extensions?.mwsVaisProductVideoConfig?.videoScript
        },

        videoScriptTemplateId() {
            return this.videoScript?.meta?.templateId
        },

        totalSceneDuration() {
            if(!this.hasVideoScript) return 0;

            const duration = this.videoScript.scenes.reduce((sum, scene) => {
                return sum + (scene.sceneScriptOutputValue ? scene.sceneScriptOutputValue.length / 16 : 0);
            }, 0);

            return Math.floor(duration);
        },

        estimatedDuration() {
            return formatDuration(this.totalSceneDuration);
        }
    },

    watch: {
        product() {
            this.key = new Date()
        }
    },

    methods: {
        async generateVideo() {
            try {
                this.pending = true

                if (!(await saveProduct('product.save.video-control'))) {
                    return;
                }

                const response = await this.MwsVaisAiVideoManager.generate(this.product.id)
                this.product.extensions.mwsVaisProductVideoConfig.generateVideoState = 'pending'

                this.createNotificationSuccess({
                    title: this.$tc('mwsvais.product-detail.video-control.generate-video'),
                    message: this.$tc(response.message)
                })
            } catch (error) {
                this.product.extensions.mwsVaisProductVideoConfig.generateVideoState = 'error'

                this.createNotificationError({
                    title: this.$tc('mwsvais.product-detail.field.video-script-button.label'),
                    message: error.response?.data?.message
                });
            } finally {
                this.pending = false
            }
        },

        async publishVideo(aiVideoId) {
            try {
                this.isPublishingVideo = true

                const aiVideo = this.aiVideos.find(item => item.id === aiVideoId)

                try {
                    await this.MwsVaisAiVideoManager.publishVideo(
                        aiVideo.id
                    )
                    this.aiVideos = await this.fetchAiVideos()

                    this.createNotificationSuccess({
                        title: this.$tc('mwsvais.product-detail.video-control.publishVideoNotification.success.title'),
                        message: this.$tc('mwsvais.product-detail.video-control.publishVideoNotification.success.message')
                    })
                } catch (e) {
                    console.error(e)
                }
            } finally {
                this.isPublishingVideo = false
            }
        },

        async unpublishVideo(aiVideoId) {
            try {
                this.isPublishingVideo = true

                const aiVideo = this.aiVideos.find(item => item.id === aiVideoId)

                try {
                    await this.MwsVaisAiVideoManager.unpublishVideo(
                        aiVideo.id
                    )
                    this.aiVideos = await this.fetchAiVideos()

                    this.createNotificationSuccess({
                        title: this.$tc('mwsvais.product-detail.video-control.publishVideoNotification.success.title'),
                        message: this.$tc('mwsvais.product-detail.video-control.unpublishVideoNotification.success.message')
                    })
                } catch (e) {
                    console.error(e)
                }
            } finally {
                this.isPublishingVideo = false
            }
        },

        async fetchAiVideos() {
            const criteria = new Criteria();
            criteria.addFilter(
                Criteria.equals('productId', this.product.id)
            );
            /*criteria.addFilter(
                Criteria.equals('languageId', Shopware.Context.api.languageId)
            )*/ // TODO: Add when language is implemented in middleware
            criteria.addAssociation('media')

            criteria.addSorting(
                Criteria.sort('createdAt', 'DESC')
            )

            criteria.setLimit(1)

            return await this.aiVideoRepository.search(criteria, Shopware.Context.Api)
        },

        async loadVideoTemplates() {
            this.isLoading = true;

            const response = await this.fetchVideoTemplates.getVideoTemplates();

            this.videoTemplates = {}
            response.templates.forEach((template) => {
                this.videoTemplates[template.id] = template.title
            })
        }
    }
})