import template from './template.html.twig'
import './style.scss'
import {ALLOWED_IMAGE_MIME_TYPES} from '@MwsVais/util/image'

const {mapState} = Shopware.Component.getComponentHelper();

Shopware.Component.register("sw-product-detail-ai-video", {
    template,
    inject: [
        'fetchVideoTemplates',
        'loginService',
        'repositoryFactory',
        'MwsVaisProductMediaService'
    ],
    data() {
        return {
            videoTemplates: [],
            formOfAddressOptions: [{value: 'Du', name: 'Du'}, {value: 'Sie', name: 'Sie'}],
            systemConfigService: Shopware.Service('systemConfigApiService'),
            pluginVideoSettings: null,
            selectedMedia: [],
            columnCount: 8,
            isLoading: true
        }
    },
    computed: {
        ...mapState('swProductDetail', [
            'product'
        ]),

        productImages() {
            return (this.product?.media ?? []).filter((item) => {
                return ALLOWED_IMAGE_MIME_TYPES.includes(item.media.mimeType)
            })
        },

        mediaItems() {
            return this.productImages.map((item) => {
                const isSelected = this.selectedMedia.includes(item.id)

                return {
                    ...item,
                    selected: isSelected
                };
            });
        },

        selectedMediaEntities() {
            if (this.selectedMedia.length < 1) {
                return this.productImages
            }

            return this.mediaItems
                .filter((item) => item.selected === true)
                .map((item) => item)
        },

        productSettingRepository() {
            return this.repositoryFactory.create('mws_vais_product_video_config')
        },

        productSettingMediaRepository() {
            return this.repositoryFactory.create('mws_vais_product_video_config_media')
        },

        isMediaSelected() {
            return (mediaId) => {
                return this.selectedMedia.includes(mediaId);
            };
        },
        useProductMedia() {
            if (this.product?.extensions?.mwsVaisProductVideoConfig?.useProductImages == null) {
                return this.pluginVideoSettings?.useProductImages
            }

            return this.product.extensions.mwsVaisProductVideoConfig.useProductImages
        },
    },
    async created() {
        await this.loadVideoTemplates();
        await this.loadPluginConfig();
        await this.createProductExtension();
        await this.loadSelectedMedia();
        await this.initSelectedMedia();
        this.isLoading = false
    },
    methods: {
        async loadVideoTemplates() {
            try {
                const response = await this.fetchVideoTemplates.getVideoTemplates();

                this.videoTemplates = response.templates.map(template => ({
                    id: (template.id).toString(),
                    label: template.title
                }));
            } catch (error) {
                console.error('Error while loading video templates:', error);
            }
        },

        async loadPluginConfig() {
            try {
                const config = await this.systemConfigService.getValues('VideoAISoftware.config');

                this.pluginVideoSettings = {
                    videoTemplateId: config['VideoAISoftware.config.videoTemplateId'] ?? null,
                    useProductName: config['VideoAISoftware.config.useProductName'] ?? false,
                    useProductDescription: config['VideoAISoftware.config.useProductDescription'] ?? false,
                    useProductProperties: config['VideoAISoftware.config.useProductProperties'] ?? false,
                    useManufacturerInformation: config['VideoAISoftware.config.useManufacturerInformation'] ?? false,
                    useProductPrice: config['VideoAISoftware.config.useProductPrice'] ?? false,
                    useProductImages: config['VideoAISoftware.config.useProductImages'] ?? false,
                    removeBackgroundAutomatically: config['VideoAISoftware.config.removeBackgroundAutomatically'] ?? false,
                    formOfAddress: config['VideoAISoftware.config.formOfAddress'] ?? null,
                    additionalVideoContent: null
                };
            } catch (error) {
                console.error('Error while loading plugin config:', error);
            }
        },

        async loadSelectedMedia() {
            try {
                const response = await this.MwsVaisProductMediaService.fetchSelectedMedia({productId: this.product.id});
                this.selectedMedia = response.mediaIds;
            } catch (error) {
                console.error('Error while loading media:', error)
            }
        },

        async initSelectedMedia() {
            if (!this.useProductMedia) return;
            if (this.selectedMedia.length > 0) return;

            await this.selectAllImages();
        },

        async createProductExtension() {
            if (this.product?.extensions?.mwsVaisProductVideoConfig) return;

            this.product.extensions.mwsVaisProductVideoConfig = this.productSettingRepository.create(Shopware.Context.api);

            this.product.extensions.mwsVaisProductVideoConfig.productId = this.product.id;
            this.product.extensions.mwsVaisProductVideoConfig.versionId = this.product.versionId;

            await this.productSettingRepository.save(this.product.extensions.mwsVaisProductVideoConfig, Shopware.Context.api);
        },

        async selectAllImages() {
            await Promise.all(
                this.productImages.map((media) =>
                    this.selectImage(media.id)
                )
            )
        },

        async deselectAllImages() {
            await Promise.all(
                this.productImages.map((media) =>
                    this.deselectImage(media.id)
                )
            )
        },

        async selectImage(mediaId) {
            if (this.isMediaSelected(mediaId)) return;

            await this.MwsVaisProductMediaService.saveSelectedMedia({
                productId: this.product.id,
                mediaId: mediaId,
                selected: true
            });

            this.selectedMedia.push(mediaId);
        },

        async deselectImage(mediaId) {
            if (!this.isMediaSelected(mediaId)) return;

            await this.MwsVaisProductMediaService.saveSelectedMedia({
                productId: this.product.id,
                mediaId: mediaId,
                selected: false
            })

            this.selectedMedia = this.selectedMedia.filter(media => media !== mediaId);
        },
    },

    watch: {
        async useProductMedia() {
            if (!this.isLoading) return;

            if (this.useProductMedia) {
                await this.selectAllImages()
            } else {
                await this.deselectAllImages()
            }
        },
    }
})