<?php declare(strict_types=1);

namespace VideoAISoftware\Controller\Api\AiVideoManager;

use Psr\Log\LoggerInterface;
use Shopware\Core\Content\Product\Exception\ProductNotFoundException;
use Shopware\Core\Framework\Api\Controller\ApiController;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\NotFilter;
use Shopware\Core\Framework\Uuid\Uuid;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Attribute\MapRequestPayload;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\Service\Attribute\Required;
use Throwable;
use VideoAISoftware\Component\Notification\NotificationService;
use VideoAISoftware\Component\ProductAiVideo\VideoManager;
use VideoAISoftware\Content\BulkMeta\BulkMetaDefinition;
use VideoAISoftware\Content\BulkMeta\BulkMetaEntity;
use VideoAISoftware\Content\BulkMeta\BulkState;
use VideoAISoftware\Content\ProductSetting\GenerateVideoState;
use VideoAISoftware\Content\ProductSetting\ProductSettingDefinition;
use VideoAISoftware\Exception\ParseUrlException;
use VideoAISoftware\Exception\ProductHasNoProductSettingException;
use VideoAISoftware\Exception\ProductHasNoVideoScriptException;
use VideoAISoftware\Exception\TemplateNotSetException;
use VideoAISoftware\Util\ContextHelper;
use VideoAISoftware\Util\PluginConfigService;
use VideoAISoftware\Util\ValidateWebhookSignature;

#[Route(defaults: ['_routeScope' => ['api']])]
class AiVideoManagerController extends ApiController
{
    private readonly EntityRepository $productRepository;
    private readonly EntityRepository $productSettingRepository;
    private readonly ContextHelper $contextHelper;
    private readonly EntityRepository $bulkMetaRepository;
    private VideoManager $videoManager;
    private readonly PluginConfigService $pluginConfigService;
    private readonly LoggerInterface $logger;

    #[Required]
    public function setProductRepository(EntityRepository $productRepository): void
    {
        $this->productRepository = $productRepository;
    }

    #[Required]
    public function setProductSettingRepository(
        #[Autowire(service: ProductSettingDefinition::ENTITY_NAME . '.repository')]
        EntityRepository $productSettingRepository
    ): void
    {
        $this->productSettingRepository = $productSettingRepository;
    }

    #[Required]
    public function setContextHelper(ContextHelper $contextHelper): void
    {
        $this->contextHelper = $contextHelper;
    }

    #[Required]
    public function setBulkMetaRepository(
        #[Autowire(service: BulkMetaDefinition::ENTITY_NAME . '.repository')]
        EntityRepository $bulkMetaRepository): void
    {
        $this->bulkMetaRepository = $bulkMetaRepository;
    }

    #[Required]
    public function setVideoManager(VideoManager $videoManager): void
    {
        $this->videoManager = $videoManager;
    }

    #[Required]
    public function setPluginConfigService(PluginConfigService $pluginConfigService): void
    {
        $this->pluginConfigService = $pluginConfigService;
    }

    #[Required]
    public function setLogger(#[Autowire(service: 'monolog.logger.business_events')] LoggerInterface $logger): void
    {
        $this->logger = $logger;
    }

    #[Route(path: '/api/mws-vais/ai-video-manager/generate-bulk', name: 'api.mws-vais.ai-video-manager.generate-video.bulk', methods: ['POST'])]
    public function generateVideoBulk(
        #[MapRequestPayload] GenerateVideoBulkDto $request
    ): JsonResponse
    {
        $context = $this->contextHelper->createContextForLanguage(
            $this->pluginConfigService->getProductLanguageId()
        );

        $bulkId = Uuid::randomHex();

        $this->bulkMetaRepository->create(array_map(fn(string $productId) => [
            'bulkId' => $bulkId,
            'productId' => $productId,
            'bulkState' => BulkState::PENDING->value
        ], $request->productIds), $context);

        $errors = [];

        foreach ($request->productIds as $productId) {
            try {
                $this->handleProduct($productId, $context);
                continue;
            } catch (ClientExceptionInterface|ServerExceptionInterface $exception) {
                $this->handleHttpException($productId, $exception, $context);
            } catch (Throwable $exception) {
                $this->handleException($productId, $exception, $context);
            }

            $errors[] = [
                'bulkId' => $bulkId,
                'productId' => $productId,
                'bulkState' => GenerateVideoState::ERROR->value
            ];
        }

        if (!empty($errors)) {
            $this->bulkMetaRepository->update($errors, $context);
        }

        return new JsonResponse([
            'count' => count($request->productIds) - count($errors),
            'errorCount' => count($errors),
            'total' => count($request->productIds)
        ]);
    }

    /**
     * @throws \Exception
     */
    #[Route(path: '/api/mws-vais/ai-video-manager/generate/{productId}', name: 'api.mws-vais.ai-video-manager.generate-video', methods: ['POST'])]
    public function generateVideo(
        string  $productId
    ): JsonResponse
    {
        $context = $this->contextHelper->createContextForLanguage(
            $this->pluginConfigService->getProductLanguageId()
        );

        try {
            $middlewareMessage = $this->handleProduct(
                $productId,
                $context
            );

            return new JsonResponse($middlewareMessage);
        } catch (ClientExceptionInterface|ServerExceptionInterface $exception) {
            $errorResponse = $this->handleHttpException($productId, $exception, $context);
            return new JsonResponse($errorResponse, $exception->getCode());
        } catch (Throwable $throwable) {
            $errorResponse = $this->handleException($productId, $throwable, $context);
            return new JsonResponse($errorResponse, $throwable->getCode());
        }
    }

    private function handleHttpException(
        string                                            $productId,
        ClientExceptionInterface|ServerExceptionInterface $exception,
        Context                                           $context
    ): array
    {
        $responseData = $exception->getResponse()->toArray(false);

        $this->productSettingRepository->upsert([[
            'productId' => $productId,
            'generateVideoState' => GenerateVideoState::ERROR->value,
            'generateVideoError' => $responseData
        ]], $context);

        return $responseData;
    }

    private function handleException(
        string    $productId,
        Throwable $throwable,
        Context   $context
    ): array
    {
        $responseData = [
            'message' => $throwable->getMessage(),
            'trace' => $throwable->getTrace()
        ];

        $this->productSettingRepository->upsert([[
            'productId' => $productId,
            'generateVideoState' => GenerateVideoState::ERROR->value,
            'generateVideoError' => $responseData
        ]], $context);

        return $responseData;
    }

    /**
     * @throws TemplateNotSetException
     * @throws ProductHasNoVideoScriptException
     * @throws ProductHasNoProductSettingException
     */
    private function handleProduct(
        string  $productId,
        Context $context
    ): array
    {
        return $this->videoManager->generateVideo($productId, $context);
    }

    /**
     * @throws ParseUrlException
     */
    #[Route(path: '/mws-vais/video-generate/webhook', name: 'frontend.mws-vais.ai-video-manager.generate-video.webhook', defaults: ['_routeScope' => ['storefront']], methods: ['POST'])]
    public function videoWebhook(
        #[MapRequestPayload] VideoWebhookDto $request,
        VideoManager                         $videoManager,
        NotificationService                  $notificationService,
        Request $rawRequest,
    ): JsonResponse
    {
        try {
            $productId = $request->id;

            $this->validateProductExistence($productId);

            $context = $this->contextHelper->createCLIContext();

            $signatureFromRequestHeader = $rawRequest->headers->get('Signature');
            if (!$signatureFromRequestHeader) {
                throw new AccessDeniedHttpException('Missing webhook signature');
            }
            $apiToken = $this->pluginConfigService->getApiKey();
            ValidateWebhookSignature::validate($signatureFromRequestHeader, $rawRequest->getContent(), $apiToken);

            if ($request->success !== true) {
                $this->productSettingRepository->upsert([[
                    'productId' => $productId,
                    'generateVideoState' => GenerateVideoState::ERROR->value,
                    'generateVideoError' => $request->error
                ]], $context);

                $bulkData = $this->handleBulk($request);

                $notificationService->createNotification(
                    'mws_vais.generate_video.webhook',
                    [
                        'productId' => $productId,
                        'success' => false
                    ]
                );

                return new JsonResponse(['message' => 'Successfully received', 'bulkData' => $bulkData]);
            }

            $videoManager->persist(
                url: $request->download,
                productId: $productId,
                context: $context
            );

            $this->productSettingRepository->upsert([[
                'productId' => $productId,
                'generateVideoState' => GenerateVideoState::SUCCESS->value,
            ]], $context);

            $bulkData = $this->handleBulk($request);

            $notificationService->createNotification(
                'mws_vais.generate_video.webhook',
                [
                    'productId' => $productId,
                    'success' => true,
                    'bulkData' => $bulkData
                ]
            );

            return new JsonResponse(['message' => 'Successfully received']);
        } catch (Throwable $throwable) {
            $this->logger->error("Video-Webhook failed: " . $throwable->getMessage(), [
                'request' => [
                    'id' => $request->id,
                    'success' => $request->success,
                    'download' => $request->download,
                    'error' => $request->error,
                ],
                'exception' => [
                    'file' => $throwable->getFile(),
                    'line' => $throwable->getLine(),
                    'exception-class' => get_class($throwable),
                    'trace' => $throwable->getTrace(),
                ]
            ]);
            throw $throwable;
        }
    }

    #[Route(path: '/api/mws-vais/ai-video-manager/publish/{id}', name: 'api.mws-vais.ai-video-manager.publish', methods: ['POST'])]
    public function publishVideo(
        string       $id,
        Context      $context,
        VideoManager $videoManager
    ): JsonResponse
    {
        $videoManager->publish(
            id: $id,
            context: $context
        );

        return new JsonResponse();
    }

    #[Route(path: '/api/mws-vais/ai-video-manager/unpublish/{id}', name: 'api.mws-vais.ai-video-manager.unpublish', methods: ['POST'])]
    public function unpublishVideo(
        string       $id,
        Context      $context,
        VideoManager $videoManager
    ): JsonResponse
    {
        $videoManager->unpublish(
            id: $id,
            context: $context
        );

        return new JsonResponse();
    }

    private function validateProductExistence(string $id): void
    {
        if (!$this->productRepository->searchIds(
            new Criteria([$id]),
            $this->contextHelper->createCLIContext()
        )->firstId()) {
            throw new ProductNotFoundException($id);
        }
    }

    private function getBulkMeta(string $productId, Context $context): ?BulkMetaEntity
    {
        return $this->bulkMetaRepository->search(
            (new Criteria())
                ->addFilter(
                    new EqualsFilter('productId', $productId),
                    new EqualsFilter('bulkState', BulkState::PENDING->value)
                ),
            $context
        )->first();
    }

    private function handleBulk(VideoWebhookDto $request): ?array
    {
        $context = $this->contextHelper->createCliContext();

        $bulkMeta = $this->getBulkMeta($request->id, $context);

        if (!$bulkMeta) {
            return null;
        }

        $this->bulkMetaRepository->update([[
            'bulkId' => $bulkMeta->getBulkId(),
            'productId' => $bulkMeta->getProductId(),
            'bulkState' => match ($request->success) {
                true => GenerateVideoState::SUCCESS->value,
                false => GenerateVideoState::ERROR->value
            }
        ]], $context);

        $finished = $this->bulkMetaRepository->searchIds(
            (new Criteria())
                ->addFilter(
                    new EqualsFilter('bulkId', $bulkMeta->getBulkId()),
                    new EqualsFilter('bulkState', BulkState::SUCCESS->value)
                ),
            $context
        )->getTotal();

        $total = $this->bulkMetaRepository->searchIds(
            (new Criteria())
                ->addFilter(
                    new EqualsFilter('bulkId', $bulkMeta->getBulkId()),
                    new NotFilter(NotFilter::CONNECTION_AND, [
                        new EqualsFilter('bulkState', BulkState::ERROR->value)
                    ])
                ), $context
        )->getTotal();

        return [
            'finished' => $finished,
            'total' => $total
        ];
    }
}