<?php declare(strict_types=1);

namespace VideoAISoftware\Migration;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\ParameterType;
use Shopware\Core\Framework\Log\Package;
use Shopware\Core\Framework\Migration\MigrationStep;
use Shopware\Core\Framework\Uuid\Uuid;
use VideoAISoftware\Content\ProductSetting\ProductSettingDefinition;

/**
 * @internal
 */
#[Package('core')]
class Migration1747308640UnsetUnsupportedImageVariables extends MigrationStep
{
    public function getCreationTimestamp(): int
    {
        return 1747308640;
    }

    public function update(Connection $connection): void
    {
        $result = $connection->createQueryBuilder()
            ->select([
                'lower(hex(product_id)) as id',
                'video_script'
            ])
            ->from(ProductSettingDefinition::ENTITY_NAME)
            ->executeQuery();

        $mediaSupportedQuery = $connection->prepare(<<<SQL
            SELECT COUNT(*) 
            FROM media
            where id = :id
            and mime_type in ('image/jpeg', 'image/png', 'image/svg+xml', 'image/webp')
        SQL
        );

        if ($this->isMariadb($connection)) {
            $updateStmt = $connection->prepare(<<<SQL
            UPDATE mws_vais_product_video_config
                SET video_script = JSON_SET(video_script, :jsonPath, null)
                WHERE product_id = :configId        
        SQL
            );
        } else {
            $updateStmt = $connection->prepare(<<<SQL
            UPDATE mws_vais_product_video_config
                SET video_script = JSON_SET(video_script, :jsonPath, CAST(NULL AS JSON))
                WHERE product_id = :configId        
        SQL
            );
        }

        foreach ($result->iterateAssociative() as $row) {
            if (!array_key_exists('video_script', $row) || $row['video_script'] === null) {
                continue;
            }
            $videoScript = json_decode($row['video_script'], true, JSON_THROW_ON_ERROR);
            if (!array_key_exists('scenes', $videoScript)) {
                continue;
            }

            foreach ($videoScript['scenes'] as $sceneIndex => $scene) {
                if (!array_key_exists('variables', $scene)) {
                    continue;
                }

                foreach ($scene['variables'] as $variableIndex => $variable) {
                    if (!array_key_exists('variableType', $variable)
                        || !array_key_exists('variableOutputValue', $variable)
                    ) {
                        continue;
                    }

                    if ($variable['variableType'] !== 'image') {
                        continue;
                    }

                    if (empty($variable['variableOutputValue'])) {
                        continue;
                    }

                    $mediaSupportedQuery->bindValue(':id', Uuid::fromHexToBytes($variable['variableOutputValue']), ParameterType::BINARY);

                    if ($mediaSupportedQuery->executeQuery()->fetchOne() > 0) {
                        continue;
                    }

                    $jsonPath = "$.scenes[$sceneIndex].variables[$variableIndex].variableOutputValue";
                    $updateStmt->bindValue('configId', Uuid::fromHexToBytes($row['id']), ParameterType::BINARY);
                    $updateStmt->bindValue('jsonPath', $jsonPath);
                    $updateStmt->executeStatement();
                }
            }
        }
    }

    private function isMariadb(Connection $connection): bool
    {
        $version = $connection->fetchOne('SELECT VERSION() AS version');
        return str_contains($version, 'MariaDB');
    }
}
