<?php declare(strict_types=1);

namespace VideoAISoftware\Services;

use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Error;
use Shopware\Core\Content\Product\Aggregate\ProductMedia\ProductMediaEntity;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use VideoAISoftware\Constants;
use VideoAISoftware\Content\ProductSetting\Media\ProductSettingMediaDefinition;


class ProductSettingService {


    
    public function __construct(
        #[AutoWire(service: ProductSettingMediaDefinition::ENTITY_NAME . '.repository')]
        private EntityRepository $productSettingMediaRepository,
        private EntityRepository $productRepository,
        private EntityRepository $productMediaRepository
    )
    {

    }

    public function getSelectedMedia(string $productId) {
        
        $criteria = new Criteria();

        $criteria->addFilter(new EqualsFilter('productId', $productId));

        $result = $this->productSettingMediaRepository->search($criteria, Context::createDefaultContext());

        $mediaIds = $result->getEntities()->map(function($entity) {
            return $entity->getProductMediaId();
        });

        return [
            'success' => true,
            'mediaIds' => array_values($mediaIds)
        ];
    }

    public function saveSelectedMedia(string $productId, string $mediaId, bool $selected) {
       
        $context = Context::createDefaultContext();

        $product = $this->validateProductId($context, $productId);
        $productMedia = $this->validateProductMediaId($context, $mediaId);

        if($selected) {
            $this->addProductMedia($context, $product, $productMedia);
        } else {
            $this->removeProductMedia($context, $product, $productMedia);
        }

        return ['success' => true];
    }

    private function validateProductId(Context $context, string $productId) : ProductEntity {
        $criteria = new Criteria();

        $criteria->addFilter(new EqualsFilter('id', $productId));

        $products = $this->productRepository->search($criteria, $context);

        if($products->count() == 0) {
            throw new Error("Invalid product id");
        }

        return $products->first();
    }

    private function validateProductMediaId(Context $context, string $mediaId) : ProductMediaEntity {
        
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('id', $mediaId));
        $criteria->addFilter(new EqualsAnyFilter('media.mimeType', Constants::ALLOWED_IMAGE_MIME_TYPES));
    
        $productMedia = $this->productMediaRepository->search($criteria, $context);
    
        if($productMedia->count() == 0) {
            throw new Error("Invalid product media id");
        }

        return $productMedia->first();
    }

    private function addProductMedia(Context $context, ProductEntity $product, ProductMediaEntity $productMedia) {

        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('productId', $product->getId()));
        $criteria->addFilter(new EqualsFilter('productMediaId', $productMedia->getId()));

        $media = $this->productSettingMediaRepository->search($criteria, $context);

        if($media->count() === 0) {

            $data = [
                'productId' => $product->getId(),
                'productVersionId' => $product->getVersionId(),
                'productMediaId' => $productMedia->getId()
            ];

            $this->productSettingMediaRepository->upsert([$data], $context);
        }

    }

    private function removeProductMedia(Context $context, ProductEntity $product, ProductMediaEntity $productMedia) {
        
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('productId', $product->getId()));
        $criteria->addFilter(new EqualsFilter('productMediaId', $productMedia->getId()));

        $media = $this->productSettingMediaRepository->search($criteria, $context);

        if($media->count() !== 0) {
            $result = $this->productSettingMediaRepository->delete([
                [ 
                    'productId' => $product->getId(),
                    'productMediaId' => $productMedia->getId()
                ]
            ], $context);
        }
    }
}